/*
 * Decompiled with CFR 0.152.
 */
package org.ethereum.db.prune;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.ethereum.crypto.HashUtil;
import org.ethereum.datasource.CountingQuotientFilter;
import org.ethereum.datasource.JournalSource;
import org.ethereum.datasource.QuotientFilter;
import org.ethereum.datasource.Source;
import org.ethereum.db.prune.Chain;
import org.ethereum.db.prune.Segment;
import org.ethereum.util.ByteArraySet;
import org.ethereum.util.ByteUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Pruner {
    private static final Logger logger = LoggerFactory.getLogger((String)"prune");
    Source<byte[], JournalSource.Update> journal;
    Source<byte[], ?> storage;
    QuotientFilter filter;
    QuotientFilter distantFilter;
    boolean ready = false;
    Stats maxLoad = new Stats();
    Stats maxCollisions = new Stats();
    int maxKeysInMemory = 0;
    int statsTracker = 0;
    Stats distantMaxLoad = new Stats();
    Stats distantMaxCollisions = new Stats();
    private static final int FILTER_ENTRIES_FORK = 8192;
    private static final int FILTER_ENTRIES_DISTANT = 2048;
    private static final int FILTER_MAX_SIZE = 0x3FFFFFFF;

    public Pruner(Source<byte[], JournalSource.Update> journal, Source<byte[], ?> storage) {
        this.storage = storage;
        this.journal = journal;
    }

    public boolean isReady() {
        return this.ready;
    }

    public boolean init(List<byte[]> forkWindow, int sizeInBlocks) {
        if (this.ready) {
            return true;
        }
        if (!forkWindow.isEmpty() && this.journal.get(forkWindow.get(0)) == null) {
            logger.debug("pruner init aborted: can't fetch update " + ByteUtil.toHexString(forkWindow.get(0)));
            return false;
        }
        QuotientFilter filter = this.instantiateFilter(sizeInBlocks, 8192);
        for (byte[] hash : forkWindow) {
            JournalSource.Update update = this.journal.get(hash);
            if (update == null) {
                logger.debug("pruner init aborted: can't fetch update " + ByteUtil.toHexString(hash));
                continue;
            }
            update.getInsertedKeys().forEach(filter::insert);
        }
        this.filter = filter;
        this.ready = true;
        return true;
    }

    public boolean withSecondStep() {
        return this.distantFilter != null;
    }

    public void withSecondStep(List<byte[]> mainChainWindow, int sizeInBlocks) {
        if (!this.ready) {
            return;
        }
        QuotientFilter filter = this.instantiateFilter(sizeInBlocks, 2048);
        if (!mainChainWindow.isEmpty()) {
            byte[] hash;
            JournalSource.Update update;
            int i;
            for (i = mainChainWindow.size() - 1; i >= 0 && (update = this.journal.get(hash = mainChainWindow.get(i))) != null; --i) {
                update.getInsertedKeys().forEach(filter::insert);
            }
            logger.debug("distant filter initialized with set of " + (i < 0 ? mainChainWindow.size() : mainChainWindow.size() - i) + " hashes, last hash " + ByteUtil.toHexString(mainChainWindow.get(i < 0 ? 0 : i)));
        } else {
            logger.debug("distant filter initialized with empty set");
        }
        this.distantFilter = filter;
    }

    private QuotientFilter instantiateFilter(int blocksCnt, int entries) {
        int size = Math.min(entries * blocksCnt, 0x3FFFFFFF);
        return CountingQuotientFilter.create(size, size);
    }

    public boolean init(byte[] ... upcoming) {
        return this.init(Arrays.asList(upcoming), 192);
    }

    public void feed(JournalSource.Update update) {
        if (this.ready) {
            update.getInsertedKeys().forEach(this.filter::insert);
        }
    }

    public void prune(Segment segment) {
        if (!this.ready) {
            return;
        }
        assert (segment.isComplete());
        logger.trace("prune " + segment);
        long t = System.currentTimeMillis();
        Pruning pruning = new Pruning();
        segment.forks.sort((f1, f2) -> Long.compare(f1.startNumber(), f2.startNumber()));
        segment.forks.forEach(x$0 -> pruning.revert((Chain)x$0));
        for (Chain chain : segment.forks) {
            chain.getHashes().forEach(this.journal::delete);
        }
        int nodesPostponed = 0;
        if (this.withSecondStep()) {
            nodesPostponed = this.postpone(segment.main);
        } else {
            pruning.nodesDeleted += this.persist(segment.main);
            segment.main.getHashes().forEach(this.journal::delete);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("nodes {}, keys in mem: {}, filter load: {}/{}: {}, distinct collisions: {}", new Object[]{this.withSecondStep() ? "postponed: " + nodesPostponed : "deleted: " + pruning.nodesDeleted, pruning.insertedInForks.size() + pruning.insertedInMainChain.size(), ((CountingQuotientFilter)this.filter).getEntryNumber(), ((CountingQuotientFilter)this.filter).getMaxInsertions(), String.format("%.4f", (double)((CountingQuotientFilter)this.filter).getEntryNumber() / (double)((CountingQuotientFilter)this.filter).getMaxInsertions()), ((CountingQuotientFilter)this.filter).getCollisionNumber()});
        }
        if (logger.isDebugEnabled()) {
            int collisions = ((CountingQuotientFilter)this.filter).getCollisionNumber();
            double load = (double)((CountingQuotientFilter)this.filter).getEntryNumber() / (double)((CountingQuotientFilter)this.filter).getMaxInsertions();
            if (collisions > this.maxCollisions.collisions) {
                this.maxCollisions.collisions = collisions;
                this.maxCollisions.load = load;
                this.maxCollisions.deleted = pruning.nodesDeleted;
            }
            if (load > this.maxLoad.load) {
                this.maxLoad.load = load;
                this.maxLoad.collisions = collisions;
                this.maxLoad.deleted = pruning.nodesDeleted;
            }
            this.maxKeysInMemory = Math.max(this.maxKeysInMemory, pruning.insertedInForks.size() + pruning.insertedInMainChain.size());
            if (++this.statsTracker % 100 == 0) {
                logger.debug("fork filter: max load: " + this.maxLoad);
                logger.debug("fork filter: max collisions: " + this.maxCollisions);
                logger.debug("fork filter: max keys in mem: " + this.maxKeysInMemory);
            }
        }
        logger.trace(segment + " pruned in {}ms", (Object)(System.currentTimeMillis() - t));
    }

    public void persist(byte[] hash) {
        if (!this.ready || !this.withSecondStep()) {
            return;
        }
        logger.trace("persist [{}]", (Object)ByteUtil.toHexString(hash));
        long t = System.currentTimeMillis();
        JournalSource.Update update = this.journal.get(hash);
        if (update == null) {
            logger.debug("skip [{}]: can't fetch update", (Object)HashUtil.shortHash(hash));
            return;
        }
        int nodesDeleted = 0;
        for (byte[] key : update.getDeletedKeys()) {
            if (this.filter.maybeContains(key) || this.distantFilter.maybeContains(key)) continue;
            ++nodesDeleted;
            this.storage.delete(key);
        }
        update.getInsertedKeys().forEach(this.distantFilter::remove);
        this.journal.delete(hash);
        if (logger.isDebugEnabled()) {
            int collisions = ((CountingQuotientFilter)this.distantFilter).getCollisionNumber();
            double load = (double)((CountingQuotientFilter)this.distantFilter).getEntryNumber() / (double)((CountingQuotientFilter)this.distantFilter).getMaxInsertions();
            if (collisions > this.distantMaxCollisions.collisions) {
                this.distantMaxCollisions.collisions = collisions;
                this.distantMaxCollisions.load = load;
                this.distantMaxCollisions.deleted = nodesDeleted;
            }
            if (load > this.distantMaxLoad.load) {
                this.distantMaxLoad.load = load;
                this.distantMaxLoad.collisions = collisions;
                this.distantMaxLoad.deleted = nodesDeleted;
            }
            if (this.statsTracker % 100 == 0) {
                logger.debug("distant filter: max load: " + this.distantMaxLoad);
                logger.debug("distant filter: max collisions: " + this.distantMaxCollisions);
            }
        }
        if (logger.isTraceEnabled()) {
            logger.trace("[{}] persisted in {}ms: {}/{} ({}%) nodes deleted, filter load: {}/{}: {}, distinct collisions: {}", new Object[]{HashUtil.shortHash(hash), System.currentTimeMillis() - t, nodesDeleted, update.getDeletedKeys().size(), nodesDeleted * 100 / update.getDeletedKeys().size(), ((CountingQuotientFilter)this.distantFilter).getEntryNumber(), ((CountingQuotientFilter)this.distantFilter).getMaxInsertions(), String.format("%.4f", (double)((CountingQuotientFilter)this.distantFilter).getEntryNumber() / (double)((CountingQuotientFilter)this.distantFilter).getMaxInsertions()), ((CountingQuotientFilter)this.distantFilter).getCollisionNumber()});
        }
    }

    private int postpone(Chain chain) {
        if (logger.isTraceEnabled()) {
            logger.trace("<~ postponing " + chain + ": " + this.strSample(chain.getHashes()));
        }
        int nodesPostponed = 0;
        for (byte[] hash : chain.getHashes()) {
            JournalSource.Update update = this.journal.get(hash);
            if (update == null) {
                logger.debug("postponing: can't fetch update " + ByteUtil.toHexString(hash));
                continue;
            }
            update.getInsertedKeys().forEach(this.distantFilter::insert);
            update.getInsertedKeys().forEach(this.filter::remove);
            nodesPostponed += update.getDeletedKeys().size();
        }
        return nodesPostponed;
    }

    private int persist(Chain chain) {
        if (logger.isTraceEnabled()) {
            logger.trace("<~ persisting " + chain + ": " + this.strSample(chain.getHashes()));
        }
        int nodesDeleted = 0;
        for (byte[] hash : chain.getHashes()) {
            JournalSource.Update update = this.journal.get(hash);
            if (update == null) {
                logger.debug("pruning aborted: can't fetch update of main chain " + ByteUtil.toHexString(hash));
                return 0;
            }
            for (byte[] key : update.getDeletedKeys()) {
                if (this.filter.maybeContains(key)) continue;
                ++nodesDeleted;
                this.storage.delete(key);
            }
            update.getInsertedKeys().forEach(this.filter::remove);
        }
        return nodesDeleted;
    }

    private String strSample(Collection<byte[]> hashes) {
        Object sample = hashes.stream().limit(3L).map(HashUtil::shortHash).collect(Collectors.joining(", "));
        if (hashes.size() > 3) {
            sample = (String)sample + ", ... (" + hashes.size() + " total)";
        }
        return sample;
    }

    private class Pruning {
        Set<byte[]> insertedInMainChain = new ByteArraySet();
        Set<byte[]> insertedInForks = new ByteArraySet();
        int nodesDeleted = 0;

        private Pruning() {
        }

        private void revert(Chain chain) {
            if (logger.isTraceEnabled()) {
                logger.trace("<~ reverting " + chain + ": " + Pruner.this.strSample(chain.getHashes()));
            }
            for (byte[] hash : chain.getHashes()) {
                JournalSource.Update update = Pruner.this.journal.get(hash);
                if (update == null) {
                    logger.debug("reverting chain " + chain + " aborted: can't fetch update " + ByteUtil.toHexString(hash));
                    return;
                }
                update.getInsertedKeys().forEach(Pruner.this.filter::remove);
                update.getDeletedKeys().forEach(key -> {
                    if (!ByteUtil.isContainedTheBytesSet(this.insertedInForks, key)) {
                        this.insertedInMainChain.add((byte[])key);
                    }
                });
                update.getInsertedKeys().forEach(key -> {
                    if (!ByteUtil.isContainedTheBytesSet(this.insertedInMainChain, key)) {
                        this.insertedInForks.add((byte[])key);
                    }
                });
                for (byte[] key2 : update.getInsertedKeys()) {
                    if (Pruner.this.filter.maybeContains(key2) || ByteUtil.isContainedTheBytesSet(this.insertedInMainChain, key2)) continue;
                    ++this.nodesDeleted;
                    Pruner.this.storage.delete(key2);
                }
            }
        }
    }

    private static class Stats {
        int collisions = 0;
        int deleted = 0;
        double load = 0.0;

        private Stats() {
        }

        public String toString() {
            return String.format("load %.4f, collisions %d, deleted %d", this.load, this.collisions, this.deleted);
        }
    }
}

